import math
import time
import os
import msvcrt

# --- HDGL Machine ---
class HDGLMachine:
    def __init__(self):
        # Each channel = (name, value, rotation speed)
        self.channels = [
            ['phi', 1.6180339887, 0.02],
            ['phi_phi', 2.6180339887, 0.015],
            ['P3', 4.2360679775, 0.01],
            ['P7', 29.0344465435, 0.005]
        ]
        self.recursion_active = False
        self.offset = 2
        self.zoom = 0.22
        self.tilt = 0.0
        self.pan = 0.0

    def toggle_recursion(self):
        self.recursion_active = not self.recursion_active

    def compute_state(self, t, idx, ch_value):
        val = ch_value * math.sin(t*ch_value)
        if self.recursion_active:
            val *= 1.2  # recursion scaling
        return val * (-1 if idx % 2 else 1)

# --- Terminal 3D Hologram ---
class TerminalHologram3D:
    def __init__(self, machine, width=100, height=30):
        self.machine = machine
        self.width = width
        self.height = height

    def clear(self):
        os.system('cls' if os.name=='nt' else 'clear')

    def rotate_point(self, x, y, z):
        cos_t = math.cos(self.machine.tilt)
        sin_t = math.sin(self.machine.tilt)
        y2 = y*cos_t - z*sin_t
        z2 = y*sin_t + z*cos_t
        cos_p = math.cos(self.machine.pan)
        sin_p = math.sin(self.machine.pan)
        x2 = x*cos_p + z2*sin_p
        z3 = -x*sin_p + z2*cos_p
        return x2, y2, z3

    def render(self, t):
        grid = [[' ' for _ in range(self.width)] for _ in range(self.height)]
        base_radius = min(self.width, self.height)//6

        for ring_idx, (name, ch_value, rot_speed) in enumerate(self.machine.channels):
            dots = 360
            for i in range(dots):
                angle = math.radians(i)
                # Multi-stranded helix: rotation per channel
                phase = t * rot_speed + i*0.01
                helix_radius = base_radius + self.machine.compute_state(t, i, ch_value) + ring_idx*3
                thickness = 2 + abs(self.machine.compute_state(t, i, ch_value)/2)

                x = (helix_radius + thickness*math.cos(t+i)) * math.cos(angle)
                y = (helix_radius + thickness*math.sin(t+i)) * math.sin(angle)
                z = (i%20 - 10) + ring_idx*3

                x_rot, y_rot, z_rot = self.rotate_point(x, y, z)
                col = int(self.width/2 + x_rot*self.machine.zoom)
                row = int(self.height/2 - y_rot*self.machine.zoom)

                if 0 <= row < self.height and 0 <= col < self.width:
                    grid[row][col] = '.'

                # stereoscopic offset
                col_offset = int(col + self.machine.offset)
                if 0 <= row < self.height and 0 <= col_offset < self.width:
                    grid[row][col_offset] = '.'

        self.clear()
        print("\n".join("".join(row) for row in grid))
        print(f"Recursion: {'ON' if self.machine.recursion_active else 'OFF'} | Offset: {self.machine.offset} | Zoom: {self.machine.zoom:.2f} | Tilt: {self.machine.tilt:.2f} | Pan: {self.machine.pan:.2f}")
        print("Controls: r=toggle recursion, [/] offset, +/- zoom, w/s tilt, a/d pan, q=quit")

# --- Main Loop ---
def main():
    machine = HDGLMachine()
    renderer = TerminalHologram3D(machine, width=120, height=40)
    t = 0.0
    dt = 0.05

    while True:
        renderer.render(t)
        t += dt
        time.sleep(dt)

        if msvcrt.kbhit():
            key = msvcrt.getwch()
            if key.lower() == 'q':
                break
            elif key.lower() == 'r':
                machine.toggle_recursion()
            elif key == '+':
                machine.zoom = min(5.0, machine.zoom*1.1)   # zoom in
            elif key == '-':
                machine.zoom = max(0.05, machine.zoom/1.1) # zoom out
            elif key == '[':
                machine.offset = max(0, machine.offset-1)
            elif key == ']':
                machine.offset += 1
            elif key.lower() == 'w':
                machine.tilt += 0.05
            elif key.lower() == 's':
                machine.tilt -= 0.05
            elif key.lower() == 'a':
                machine.pan -= 0.05
            elif key.lower() == 'd':
                machine.pan += 0.05

if __name__ == "__main__":
    main()
